#!/bin/bash

# Installation script for language-switcher v0.13.2
# Content Hash: 3eb16b4ae122a2fa

set -e

# Base URL for component downloads
DOWNLOAD_BASE_URL="https://metalsmith-components.com/downloads"

echo "🔧 Installing language-switcher v0.13.2..."

# Detect Metalsmith project directory and component source
COMPONENT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"

# Check if PROJECT_ROOT was set by bundle installer
if [ -n "$PROJECT_ROOT" ]; then
  # Called from bundle installer, PROJECT_ROOT already set
  cd "$PROJECT_ROOT"
# Check if we're in a component package directory (has x-metalsmith-component marker)
elif [ -f "package.json" ] && grep -q "\"x-metalsmith-component\"" package.json; then
  # In a component directory, look up for project root
  if [ -f "../metalsmith.js" ]; then
    PROJECT_ROOT="$(cd .. && pwd)"
    cd "$PROJECT_ROOT"
  elif [ -f "../../metalsmith.js" ]; then
    # Two levels up (e.g., bundle/partials/component/)
    PROJECT_ROOT="$(cd ../.. && pwd)"
    cd "$PROJECT_ROOT"
  else
    echo "❌ Error: Component directory not in a Metalsmith project"
    echo "Please extract components in your Metalsmith project root directory"
    exit 1
  fi
elif [ -f "metalsmith.js" ]; then
  # Already in project root
  PROJECT_ROOT="$(pwd)"
else
  echo "❌ Error: Not a Metalsmith project directory"
  echo "Please run this script from your Metalsmith project root or from within the extracted component directory"
  exit 1
fi

# Load component paths from config or use defaults
if [ -f "metalsmith-components.config.json" ]; then
  COMPONENTS_BASE=$(node -p "try { require('./metalsmith-components.config.json').componentsBasePath } catch(e) { 'lib/layouts/components' }")
  SECTIONS_DIR=$(node -p "try { require('./metalsmith-components.config.json').sectionsDir } catch(e) { 'sections' }")
  PARTIALS_DIR=$(node -p "try { require('./metalsmith-components.config.json').partialsDir } catch(e) { '_partials' }")
else
  COMPONENTS_BASE="lib/layouts/components"
  SECTIONS_DIR="sections"
  PARTIALS_DIR="_partials"
fi

# Track installed dependencies to prevent circular loops
if [ -z "$INSTALLED_DEPS" ]; then
  export INSTALLED_DEPS=""
fi

# Function to download and install a dependency
install_dependency() {
  local dep_name="$1"
  local dep_type="$2"  # "partial" or "section"
  
  # Check if already processed in this session (circular dependency protection)
  if [[ "$INSTALLED_DEPS" == *":$dep_name:"* ]]; then
    return 0
  fi
  
  # Mark as being processed
  export INSTALLED_DEPS="$INSTALLED_DEPS:$dep_name:"
  
  local dep_dir
  local download_url
  
  if [ "$dep_type" = "section" ]; then
    dep_dir="$COMPONENTS_BASE/$SECTIONS_DIR/$dep_name"
    download_url="$DOWNLOAD_BASE_URL/sections/$dep_name.zip"
  else
    dep_dir="$COMPONENTS_BASE/$PARTIALS_DIR/$dep_name"
    download_url="$DOWNLOAD_BASE_URL/partials/$dep_name.zip"
  fi
  
  # Check if already installed
  if [ -f "$dep_dir/manifest.json" ]; then
    echo "  ✓ $dep_name (already installed)"
    return 0
  fi
  
  echo "  ↓ Installing $dep_name..."
  
  # Create temp directory for download
  local temp_dir=$(mktemp -d)
  local zip_file="$temp_dir/$dep_name.zip"
  
  # Download the component
  if ! curl -sL -f "$download_url" -o "$zip_file" 2>/dev/null; then
    echo "    ⚠ Failed to download $dep_name from $download_url"
    rm -rf "$temp_dir"
    return 1
  fi
  
  # Extract and install
  if ! unzip -q "$zip_file" -d "$temp_dir" 2>/dev/null; then
    echo "    ⚠ Failed to extract $dep_name"
    rm -rf "$temp_dir"
    return 1
  fi
  
  # Run the dependency's install script (handles nested dependencies)
  local extracted_dir="$temp_dir/$dep_name"
  if [ -f "$extracted_dir/install.sh" ]; then
    chmod +x "$extracted_dir/install.sh"
    # Run with AUTO_INSTALL to skip prompts
    (cd "$extracted_dir" && AUTO_INSTALL=1 ./install.sh) || {
      echo "    ⚠ Failed to install $dep_name"
      rm -rf "$temp_dir"
      return 1
    }
  fi
  
  # Cleanup
  rm -rf "$temp_dir"
  
  # Track that this was auto-installed
  AUTO_INSTALLED_DEPS="$AUTO_INSTALLED_DEPS $dep_name"
  
  return 0
}

# Create target directory
if [ "partial" = "section" ]; then
  TARGET_DIR="$COMPONENTS_BASE/$SECTIONS_DIR/language-switcher"
else
  TARGET_DIR="$COMPONENTS_BASE/$PARTIALS_DIR/language-switcher"
fi
mkdir -p "$TARGET_DIR"

# Check for existing installation
if [ -f "$TARGET_DIR/manifest.json" ]; then
  EXISTING_HASH=$(grep -o '"contentHash": "[^"]*"' "$TARGET_DIR/manifest.json" | cut -d'"' -f4)
  if [ "$EXISTING_HASH" = "3eb16b4ae122a2fa" ]; then
    echo "✓ language-switcher v0.13.2 already installed (no changes)"
    exit 0
  else
    echo "📦 Upgrading language-switcher (content changed)"
  fi
fi

# Check and auto-install dependencies
echo "Checking dependencies..."
AUTO_INSTALLED_DEPS=""
FAILED_DEPS=""

# Check for icon
if [ ! -f "$COMPONENTS_BASE/$PARTIALS_DIR/icon/manifest.json" ] && [ ! -f "$COMPONENTS_BASE/$SECTIONS_DIR/icon/manifest.json" ]; then
  # Try to auto-install (most dependencies are partials)
  if ! install_dependency "icon" "partial"; then
    # Try as section if partial fails
    if ! install_dependency "icon" "section"; then
      FAILED_DEPS="$FAILED_DEPS icon"
    fi
  fi
else
  echo "  ✓ icon (already installed)"
fi

# Check if any dependencies failed
if [ -n "$FAILED_DEPS" ]; then
  echo ""
  echo "⚠ Warning: Could not install some dependencies:$FAILED_DEPS"
  echo ""
  echo "You may need to download them manually from:"
  echo "  https://metalsmith-components.com/downloads/"
  echo ""
  # Skip interactive prompt if called from bundle installer or auto-install mode
  if [ -n "$BUNDLE_INSTALL" ] || [ -n "$AUTO_INSTALL" ]; then
    echo "  (Auto-continuing)"
    echo ""
  else
    read -p "Continue installation anyway? (y/n) " -n 1 -r
    echo
    if [[ ! $REPLY =~ ^[Yy]$ ]]; then
      exit 1
    fi
  fi
fi

# Copy files
echo "Copying files..."
cp "$COMPONENT_DIR/language-switcher.njk" "$TARGET_DIR/"
cp "$COMPONENT_DIR/language-switcher.css" "$TARGET_DIR/"
cp "$COMPONENT_DIR/language-switcher.js" "$TARGET_DIR/"
cp "$COMPONENT_DIR/manifest.json" "$TARGET_DIR/"
if [ -f "$COMPONENT_DIR/README.md" ]; then
  cp "$COMPONENT_DIR/README.md" "$TARGET_DIR/"
fi

echo ""
echo "✓ Installation complete"
echo ""
echo "Files installed to: $TARGET_DIR"
if [ -n "$AUTO_INSTALLED_DEPS" ]; then
  echo "Dependencies installed:$AUTO_INSTALLED_DEPS"
fi
echo ""
# Cleanup extracted component directory if not called from bundle or auto-install
if [ -z "$BUNDLE_INSTALL" ] && [ -z "$AUTO_INSTALL" ] && [ -f "$COMPONENT_DIR/package.json" ] && grep -q "\"x-metalsmith-component\"" "$COMPONENT_DIR/package.json" 2>/dev/null; then
  # Check if component directory is in project root (not in a bundle structure)
  COMPONENT_BASENAME="$(basename "$COMPONENT_DIR")"
  if [ "$COMPONENT_DIR" = "$PROJECT_ROOT/$COMPONENT_BASENAME" ]; then
    echo ""
    read -p "Remove extracted component directory $COMPONENT_BASENAME? (y/n) " -n 1 -r
    echo
    if [[ $REPLY =~ ^[Yy]$ ]]; then
      rm -rf "$COMPONENT_DIR"
      echo "✓ Cleaned up $COMPONENT_BASENAME"
    fi
  fi
fi

echo ""
echo "See README.md for usage instructions"
